<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\data\Pagination;
use app\models\MeterReading;
use app\models\User;
use app\models\UtilityType;
use app\models\AdminFilter;
use yii\web\Response;
use yii\helpers\CsvHelper;

class AdminController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'matchCallback' => function () {
                            // Проверяем, что пользователь администратор (по номеру телефона)
                            if (!Yii::$app->user->isGuest) {
                                $user = User::findOne(Yii::$app->user->id);
                                return $user && $user->phone === '8(000)000-00-00';
                            }
                            return false;
                        },
                    ],
                ],
                'denyCallback' => function () {
                    Yii::$app->session->setFlash('error', 'Доступ запрещен. Только для администраторов.');
                    return Yii::$app->response->redirect(['site/index']);
                },
            ],
        ];
    }

    public function actionIndex()
    {
        $filterModel = new AdminFilter();
        $query = MeterReading::find()->joinWith(['user', 'utilityType']);

        // Применяем фильтры
        if ($filterModel->load(Yii::$app->request->get()) && $filterModel->validate()) {
            if (!empty($filterModel->status)) {
                $query->andWhere(['meter_readings.status' => $filterModel->status]);
            }
            if (!empty($filterModel->utility_type_id)) {
                $query->andWhere(['meter_readings.utility_type_id' => $filterModel->utility_type_id]);
            }
            if (!empty($filterModel->date_from)) {
                $query->andWhere(['>=', 'meter_readings.created_at', $filterModel->date_from . ' 00:00:00']);
            }
            if (!empty($filterModel->date_to)) {
                $query->andWhere(['<=', 'meter_readings.created_at', $filterModel->date_to . ' 23:59:59']);
            }
        }

        // Пагинация
        $countQuery = clone $query;
        $pagination = new Pagination([
            'totalCount' => $countQuery->count(),
            'pageSize' => 20,
        ]);

        $readings = $query->orderBy(['meter_readings.created_at' => SORT_DESC])
            ->offset($pagination->offset)
            ->limit($pagination->limit)
            ->all();

        // Статистика
        $stats = $this->getStats();

        $utilityTypes = UtilityType::find()->all();

        return $this->render('index', [
            'readings' => $readings,
            'pagination' => $pagination,
            'filterModel' => $filterModel,
            'stats' => $stats,
            'utilityTypes' => $utilityTypes,
        ]);
    }

    public function actionUpdateStatus($id, $status)
    {
        $reading = MeterReading::findOne($id);
        if ($reading && in_array($status, ['new', 'approved', 'rejected'])) {
            $oldStatus = $reading->status;
            $reading->status = $status;
            if ($reading->save(false)) {
                Yii::$app->session->setFlash('success', "Статус заявки #{$id} изменен с '{$oldStatus}' на '{$status}'.");
            } else {
                Yii::$app->session->setFlash('error', 'Ошибка при обновлении статуса.');
            }
        } else {
            Yii::$app->session->setFlash('error', 'Заявка не найдена или неверный статус.');
        }
        return $this->redirect(['index']);
    }

    public function actionUsers()
    {
        $query = User::find()->where(['!=', 'phone', '8(000)000-00-00']);
        $countQuery = clone $query;
        $pagination = new Pagination([
            'totalCount' => $countQuery->count(),
            'pageSize' => 20,
        ]);

        $users = $query->orderBy(['created_at' => SORT_DESC])
            ->offset($pagination->offset)
            ->limit($pagination->limit)
            ->all();

        return $this->render('users', [
            'users' => $users,
            'pagination' => $pagination,
        ]);
    }

    public function actionExport()
    {
        $readings = MeterReading::find()
            ->joinWith(['user', 'utilityType'])
            ->orderBy(['meter_readings.created_at' => SORT_DESC])
            ->all();

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=readings_' . date('Y-m-d') . '.csv');

        $output = fopen('php://output', 'w');
        fputcsv($output, [
            'ID',
            'Дата',
            'Пользователь',
            'Телефон',
            'Услуга',
            'Предыдущие показания',
            'Текущие показания',
            'Расход',
            'Сумма',
            'Статус'
        ], ';');

        foreach ($readings as $reading) {
            fputcsv($output, [
                $reading->id,
                Yii::$app->formatter->asDate($reading->created_at, 'php:d.m.Y H:i'),
                $reading->user->full_name,
                $reading->user->phone,
                $reading->utilityType->name,
                $reading->previous_reading,
                $reading->current_reading,
                $reading->consumption,
                $reading->amount,
                $reading->status
            ], ';');
        }

        fclose($output);
        exit;
    }

    public function actionTariffs()
    {
        $utilityTypes = UtilityType::find()->all();

        if (Yii::$app->request->isPost) {
            foreach ($utilityTypes as $utility) {
                $newTariff = Yii::$app->request->post('tariff')[$utility->id] ?? null;
                if ($newTariff !== null && is_numeric($newTariff) && $newTariff > 0) {
                    $utility->tariff = $newTariff;
                    $utility->save(false);
                }
            }
            Yii::$app->session->setFlash('success', 'Тарифы успешно обновлены.');
            return $this->refresh();
        }

        return $this->render('tariffs', [
            'utilityTypes' => $utilityTypes,
        ]);
    }

    /**
     * Получение статистики для админ-панели
     */
    private function getStats()
    {
        return [
            'total_readings' => MeterReading::find()->count(),
            'new_readings' => MeterReading::find()->where(['status' => 'new'])->count(),
            'approved_readings' => MeterReading::find()->where(['status' => 'approved'])->count(),
            'rejected_readings' => MeterReading::find()->where(['status' => 'rejected'])->count(),
            'total_users' => User::find()->where(['!=', 'phone', '8(000)000-00-00'])->count(),
            'total_amount' => MeterReading::find()->where(['status' => 'approved'])->sum('amount') ?: 0,
            'last_activity' => MeterReading::find()->select('created_at')->orderBy(['created_at' => SORT_DESC])->scalar()
                ? Yii::$app->formatter->asRelativeTime(MeterReading::find()->select('created_at')->orderBy(['created_at' => SORT_DESC])->scalar())
                : 'нет данных',
        ];
    }
}