<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use app\models\MeterReading;
use app\models\UtilityType;
use yii\web\Response;

class ReadingsController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
        ];
    }

    public function actionIndex()
    {
        $readings = MeterReading::find()
            ->where(['user_id' => Yii::$app->user->id])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('index', [
            'readings' => $readings,
        ]);
    }

    public function actionCreate()
    {
        $model = new MeterReading();
        $utilityTypes = UtilityType::find()->all();

        if ($model->load(Yii::$app->request->post())) {
            // Получаем предыдущие показания для этого типа услуги и пользователя
            $lastReading = MeterReading::find()
                ->where(['user_id' => Yii::$app->user->id, 'utility_type_id' => $model->utility_type_id])
                ->orderBy(['created_at' => SORT_DESC])
                ->one();

            if ($lastReading) {
                $model->previous_reading = $lastReading->current_reading;
            } else {
                // Если это первая передача, то предыдущие показания 0
                $model->previous_reading = 0;
            }

            if ($model->validate()) {
                if ($model->save()) {
                    Yii::$app->session->setFlash('success', 'Показания успешно переданы и отправлены на проверку администратору.');
                    return $this->redirect(['index']);
                } else {
                    Yii::$app->session->setFlash('error', 'Ошибка при сохранении показаний.');
                }
            } else {
                Yii::$app->session->setFlash('error', 'Ошибка валидации: ' . implode(', ', $model->getFirstErrors()));
            }
        }

        return $this->render('create', [
            'model' => $model,
            'utilityTypes' => $utilityTypes,
        ]);
    }

    /**
     * AJAX метод для получения последних показаний
     */
    public function actionGetLastReading()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $utility_type_id = Yii::$app->request->get('utility_type_id');
        $userId = Yii::$app->user->id;

        if (!$userId) {
            return ['previousReading' => 0];
        }

        $lastReading = MeterReading::find()
            ->where(['user_id' => $userId, 'utility_type_id' => $utility_type_id])
            ->orderBy(['created_at' => SORT_DESC])
            ->one();

        return [
            'previousReading' => $lastReading ? $lastReading->current_reading : 0
        ];
    }

    /**
     * AJAX метод для получения тарифа
     */
    public function actionGetTariff()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $utility_type_id = Yii::$app->request->get('utility_type_id');

        $utility = UtilityType::findOne($utility_type_id);

        if ($utility) {
            return [
                'tariff' => (float)$utility->tariff,
                'unit' => $utility->unit
            ];
        }

        return ['tariff' => 0, 'unit' => ''];
    }
}