<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

class MeterReading extends ActiveRecord
{
    public static function tableName()
    {
        return 'meter_readings';
    }

    public function rules()
    {
        return [
            [['utility_type_id', 'current_reading'], 'required'],
            [['current_reading'], 'number', 'min' => 0],
            ['current_reading', 'validateReadings'],
            // Убираем previous_reading из обязательных, так как оно вычисляется автоматически
            ['previous_reading', 'safe'], // Разрешаем любое значение, так как поле вычисляемое
        ];
    }

    public function validateReadings($attribute, $params)
    {
        if ($this->current_reading < $this->previous_reading) {
            $this->addError($attribute, 'Текущие показания не могут быть меньше предыдущих');
        }
    }

    public function calculateConsumption()
    {
        return $this->current_reading - ($this->previous_reading ?: 0);
    }

    public function calculateAmount($tariff)
    {
        return $this->calculateConsumption() * $tariff;
    }

    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    public function getUtilityType()
    {
        return $this->hasOne(UtilityType::class, ['id' => 'utility_type_id']);
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            $utility = UtilityType::findOne($this->utility_type_id);

            // Если previous_reading не установлено, получаем из последних показаний
            if (!$this->previous_reading) {
                $lastReading = MeterReading::find()
                    ->where(['user_id' => Yii::$app->user->id, 'utility_type_id' => $this->utility_type_id])
                    ->orderBy(['created_at' => SORT_DESC])
                    ->one();

                $this->previous_reading = $lastReading ? $lastReading->current_reading : 0;
            }

            $this->consumption = $this->calculateConsumption();
            $this->amount = $this->calculateAmount($utility->tariff);
            $this->reading_date = date('Y-m-d');

            if ($insert) {
                $this->user_id = Yii::$app->user->id;
            }
            return true;
        }
        return false;
    }
}